<?php

namespace App\Http\Controllers\Admin;

use Exception;
use Carbon\Carbon;
use App\Models\Cuti;
use App\Models\User;
use App\Models\Absensi;
use App\Models\Karyawan;
use App\Models\JatahCuti;
use App\Models\Notifikasi;
use App\Models\CutiApproval;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;

class ApprovalController extends Controller
{
    /**
     * Display list approval page
     */
    public function index()
    {
        $userRole = Auth::user()->role;

        // Ambil cuti yang menunggu approval sesuai role user
        $cutiQuery = Cuti::with(['karyawan.user', 'karyawan.departemen', 'jenisCuti', 'approvals.approver'])
                    ->where('status', 'pending');

        // Filter berdasarkan role
        if (in_array($userRole, ['manager', 'gm', 'hrd'])) {
            // Manager/GM/HRD hanya lihat cuti di step mereka
            $cutiQuery->where('current_step', $userRole);
        }

        // PAGINATION: 15 items per page
        $cuti = $cutiQuery->orderBy('created_at', 'desc')->paginate(15);

        // Hitung stats
        $cutiCount = [
            'pending' => $cuti->total()
        ];

        $approvedToday = CutiApproval::where('approved_by', Auth::id())
                                     ->where('status', 'disetujui')
                                     ->whereDate('approved_at', today())
                                     ->count();

        $rejectedToday = CutiApproval::where('approved_by', Auth::id())
                                     ->where('status', 'ditolak')
                                     ->whereDate('approved_at', today())
                                     ->count();

        return view('admin.approval.index', compact(
            'cuti',
            'cutiCount',
            'approvedToday',
            'rejectedToday'
        ));
    }

    /**
     * Approve Cuti - FIXED LOGIC
     */
    public function approve(Request $request, $id)
    {
        try {
            $cuti = Cuti::with(['karyawan', 'jenisCuti', 'approvals'])->findOrFail($id);
            $userRole = Auth::user()->role;

            // VALIDASI 1: Cek status cuti
            if ($cuti->status !== 'pending') {
                return back()->with('error', 'Cuti ini sudah diproses sebelumnya');
            }

            // VALIDASI 2: Cek apakah role user sesuai dengan current_step
            if ($cuti->current_step !== $userRole) {
                return back()->with('error', 'Anda tidak berhak menyetujui cuti ini pada tahap saat ini');
            }

            // VALIDASI 3: Cek apakah user mencoba approve cuti miliknya sendiri
            if ($cuti->karyawan->user_id === Auth::id()) {
                return back()->with('error', 'Anda tidak dapat menyetujui cuti Anda sendiri');
            }

            DB::beginTransaction();

            // Ambil approval step saat ini
            $currentApproval = $cuti->approvals()
                ->where('step', $userRole)
                ->where('status', 'pending')
                ->first();

            if (!$currentApproval) {
                DB::rollBack();
                return back()->with('error', 'Step approval tidak ditemukan atau sudah diproses');
            }

            // Update approval step ini
            $currentApproval->update([
                'status' => 'disetujui',
                'approved_by' => Auth::id(),
                'catatan' => $request->catatan_admin ?? null,
                'approved_at' => now()
            ]);

            // Activity log
            if (function_exists('activity_log')) {
                activity_log(
                    'cuti',
                    'approve',
                    "Menyetujui cuti {$cuti->jenisCuti->nama} karyawan {$cuti->karyawan->user->nama} sebagai {$userRole}"
                );
            }

            // ========== PERBAIKAN LOGIC CARI NEXT STEP ==========
            // Cari step berikutnya yang masih pending
            $nextApproval = CutiApproval::where('cuti_id', $cuti->id)
                ->where('status', 'pending')
                ->where('id', '>', $currentApproval->id) // ID lebih besar = step berikutnya
                ->orderBy('id', 'asc')
                ->first();

            if ($nextApproval) {
                // ===== MASIH ADA STEP BERIKUTNYA =====
                $cuti->update([
                    'current_step' => $nextApproval->step
                ]);

                // Notifikasi ke pemohon - step disetujui
                Notifikasi::create([
                    'user_id' => $cuti->karyawan->user_id,
                    'judul' => 'Persetujuan Cuti - Update',
                    'pesan' => "Pengajuan cuti {$cuti->jenisCuti->nama} Anda telah disetujui oleh " . strtoupper($userRole) . ". Menunggu persetujuan " . strtoupper($nextApproval->step) . ".",
                    'type' => 'cuti',
                    'target_role' => 'karyawan'
                ]);

                // Notifikasi ke approver berikutnya
                Notifikasi::create([
                    'user_id' => null,
                    'judul' => 'Pengajuan Cuti Menunggu Persetujuan',
                    'pesan' => "Cuti {$cuti->jenisCuti->nama} dari {$cuti->karyawan->user->nama} menunggu persetujuan Anda.",
                    'type' => 'cuti',
                    'target_role' => $nextApproval->step
                ]);

                DB::commit();
                return back()->with('success', "Cuti disetujui. Menunggu approval dari " . strtoupper($nextApproval->step));

            } else {
                // ===== SEMUA STEP SELESAI - FINAL APPROVAL =====
                $this->finalizeCutiApproval($cuti, $userRole);

                DB::commit();
                return back()->with('success', 'Cuti berhasil disetujui sepenuhnya');
            }

        } catch (Exception $e) {
            DB::rollBack();
            Log::error('Error Approve Cuti: ' . $e->getMessage());
            return back()->with('error', 'Terjadi kesalahan: ' . $e->getMessage());
        }
    }

    /**
     * Reject Cuti
     */
    public function reject(Request $request, $id)
    {
        try {
            $request->validate([
                'catatan_admin' => 'required|string|min:10'
            ]);

            $cuti = Cuti::with(['karyawan', 'jenisCuti', 'approvals'])->findOrFail($id);
            $userRole = Auth::user()->role;

            // VALIDASI 1: Cek status
            if ($cuti->status !== 'pending') {
                return back()->with('error', 'Cuti ini sudah diproses sebelumnya');
            }

            // VALIDASI 2: Role sesuai current_step
            if ($cuti->current_step !== $userRole) {
                return back()->with('error', 'Anda tidak berhak menolak cuti ini pada tahap saat ini');
            }

            // VALIDASI 3: Bukan cuti sendiri
            if ($cuti->karyawan->user_id === Auth::id()) {
                return back()->with('error', 'Anda tidak dapat menolak cuti Anda sendiri');
            }

            DB::beginTransaction();

            // Update approval step saat ini
            $currentApproval = $cuti->approvals()
                ->where('step', $userRole)
                ->where('status', 'pending')
                ->first();

            if ($currentApproval) {
                $currentApproval->update([
                    'status' => 'ditolak',
                    'approved_by' => Auth::id(),
                    'catatan' => $request->catatan_admin,
                    'approved_at' => now()
                ]);
            }

            // Update semua step berikutnya jadi ditolak
            CutiApproval::where('cuti_id', $cuti->id)
                ->where('id', '>', $currentApproval->id)
                ->update(['status' => 'ditolak']);

            // Update status cuti menjadi ditolak
            $cuti->update([
                'status' => 'ditolak',
                'current_step' => null,
                'catatan_admin' => $request->catatan_admin
            ]);

            if (function_exists('activity_log')) {
                activity_log(
                    'cuti',
                    'reject',
                    "Menolak cuti {$cuti->jenisCuti->nama} karyawan {$cuti->karyawan->user->nama} sebagai {$userRole}"
                );
            }

            // Notifikasi ke pemohon
            $pesan = "Pengajuan cuti {$cuti->jenisCuti->nama} Anda ditolak oleh " . strtoupper($userRole) . ".";
            $pesan .= "\n\nAlasan: {$request->catatan_admin}";

            Notifikasi::create([
                'user_id' => $cuti->karyawan->user_id,
                'judul' => 'Cuti Ditolak',
                'pesan' => $pesan,
                'type' => 'cuti',
                'target_role' => 'karyawan'
            ]);

            DB::commit();

            return back()->with('success', 'Pengajuan cuti berhasil ditolak');

        } catch (Exception $e) {
            DB::rollBack();
            Log::error('Error Reject Cuti: ' . $e->getMessage());
            return back()->with('error', 'Terjadi kesalahan: ' . $e->getMessage());
        }
    }
    /**
     * Display detail approval page
     */
    public function detail($id)
    {
        $cuti = Cuti::with([
                    'karyawan.user',
                    'karyawan.departemen',
                    'karyawan.jabatan',
                    'jenisCuti',
                    'approvals.approver'
                ])
                ->findOrFail($id);

        $userRole = Auth::user()->role;

        // VALIDASI: Cek apakah user berhak melihat cuti ini
        if (in_array($userRole, ['manager', 'gm', 'hrd'])) {
            // Manager/GM/HRD hanya bisa lihat cuti yang ada di step mereka atau sudah selesai
            if ($cuti->status === 'pending' && $cuti->current_step !== $userRole) {
                // Cek apakah sudah pernah di-approve oleh role ini
                $hasApproved = $cuti->approvals()
                    ->where('step', $userRole)
                    ->where('status', '!=', 'pending')
                    ->exists();

                if (!$hasApproved) {
                    abort(403, 'Anda tidak berhak mengakses cuti ini');
                }
            }
        }

        return view('admin.approval.cuti-detail', compact('cuti'));
    }
    /**
     * Helper: Finalize Cuti Approval
     */
    private function finalizeCutiApproval($cuti, $userRole)
    {
        $mulai = Carbon::parse($cuti->tanggal_mulai)->startOfDay();
        $selesai = Carbon::parse($cuti->tanggal_selesai)->startOfDay();
        $jumlahHari = $mulai->diffInDays($selesai) + 1;

        // Kurangi jatah cuti
        $jatah = JatahCuti::where('karyawan_id', $cuti->karyawan_id)
            ->where('tahun', now()->year)
            ->first();

        if (!$jatah || $jatah->jatah < $jumlahHari) {
            throw new Exception('Jatah cuti karyawan tidak mencukupi');
        }

        $jatah->decrement('jatah', $jumlahHari);

        // Update status cuti
        $cuti->update([
            'status' => 'disetujui',
            'current_step' => null,
            'catatan_admin' => null
        ]);

        // Insert absensi CUTI per hari
        $tanggal = Carbon::parse($cuti->tanggal_mulai);
        while ($tanggal->lte($cuti->tanggal_selesai)) {
            Absensi::updateOrCreate([
                'karyawan_id' => $cuti->karyawan_id,
                'tanggal' => $tanggal->toDateString(),
            ], [
                'status' => 'cuti',
            ]);
            $tanggal->addDay();
        }

        if (function_exists('activity_log')) {
            activity_log(
                'cuti',
                'approve',
                "Menyetujui final cuti {$cuti->jenisCuti->nama} karyawan {$cuti->karyawan->user->nama} sebagai {$userRole}"
            );
        }

        // Notifikasi ke pemohon - final
        Notifikasi::create([
            'user_id' => $cuti->karyawan->user_id,
            'judul' => 'Cuti Disetujui',
            'pesan' => "Pengajuan cuti {$cuti->jenisCuti->nama} Anda telah disetujui sepenuhnya untuk periode {$cuti->tanggal_mulai->format('d/m/Y')} - {$cuti->tanggal_selesai->format('d/m/Y')}.",
            'type' => 'cuti',
            'target_role' => 'karyawan'
        ]);
    }
}
